import java.awt.Color;
import java.awt.Graphics2D;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;

import se.egy.graphics.Drawable;

public class Map implements Drawable{
	private int cols, rows;
	private int tileSize;

	private int[][] tileMap;

	/**
	 * Konstruktor
	 * Läser in filen och rutornas storlek.
	 * I textfilen skall det på 
	 * <ul>
	 * 	<li> rad 1 stå antal kolumner (x)</li>
	 * 	<li> rad 2 stå antalet rader (y)</li>
	 * 	<li> övriga rader skrivas in en rad med ett heltal för varje ruta. Separera rutorna med mellanslag.</li>
	 * <ul>
	 * 
	 * @param pathToMap Sökvägen till textfilen med rutnätet
	 * @param tileSize  Rutornas storlek i px
	 */
	public Map(String pathToMap, int tileSize) {
		this.tileSize = tileSize;

		// Lägger till slash om det saknas innan filnamn
		if(pathToMap.charAt(0) != '/')
			pathToMap = "/" + pathToMap;

		// Öppnar filen
		BufferedReader mapFile = new BufferedReader(new InputStreamReader(this.getClass().getResourceAsStream(pathToMap)));

		readMapData(mapFile);

	}

	private void readMapData(BufferedReader mapFile) {
		// Läser antal rutor i x- och y-led. >Två första raderna
		try {
			cols = Integer.parseInt(mapFile.readLine());
			rows = Integer.parseInt(mapFile.readLine());
		} catch (Exception e) {
			e.printStackTrace();
		}

		tileMap = new int[cols][rows];

		// Laddar in tileMap från filen
		for(int row = 0; row < rows; row++) {
			String rowAsString = null;
			
			try {
				rowAsString = mapFile.readLine();
			} catch (IOException e) {
				e.printStackTrace();
			}

			String[] tile = rowAsString.split(" ");
			for(int col = 0; col < cols; col++) {
				tileMap[col][row] = Integer.parseInt(tile[col]);
			}
		}
	}


	/**
	 * Renderar Rutnätet på ett grafikobjekt
	 * @param g grafikobjekt där rutnätet skall renderas
	 */
	public void draw(Graphics2D g) {

		for (int x=0;x<cols;x++) {
			for (int y=0;y<rows;y++) {
				int tile = tileMap[x][y];

				// Vilka färger som skall väljas beroende på värde i map
				switch(tile) {
				case 0: g.setColor(Color.black); break;
				case 1: g.setColor(Color.blue); break;
				case 2: g.setColor(Color.black); break;
				case 3: g.setColor(Color.gray); break;
				case 4: g.setColor(Color.lightGray); break;
				case 5: g.setColor(Color.green); break;
				default: g.setColor(Color.black); 
				}

				g.fillRect(x*tileSize,y*tileSize,tileSize,tileSize);
				
				

				// Lite fullösning för att skapa en gul cirkel på rutor med värdet 2
				if(tile == 2) {
					g.setColor(Color.yellow);
					g.fillOval(x*tileSize, y*tileSize, 8, 8);
				}
			}
		}
	}

	/**
	 * Antal rader i map-filen
	 * @return antalet rader
	 */
	public int getRows() {
		return rows;
	}

	/**
	 * Antal kolumner i map-filen
	 * @return antalet kolumner
	 */
	public int getCols() {
		return rows;
	}

	/**
	 * Storleken på varje ruta
	 * @return Storleken i px på rutorna
	 */
	public int getTileSize() {
		return tileSize;
	}

	/**
	 * Hämtar en specifik rutas värde i rutnätet
	 * 
	 * @param col rutans kolumnnummer
	 * @param row rutans radnummer
	 * @return Värdet i rutan
	 */
	public int getTile(double col, double row) {
		return tileMap[(int)col][(int)row];
	}

	/**
	 * Hämtar en specifik rutas värde i rutnätet med utgångsdpunkt från en punkt på skärmen (px).
	 * 
	 * @param x punktens position i x-led
	 * @param y punktens position i y-led
	 * 
	 * @return rutans värde i rutnätet
	 */
	public int getTileFromPx(double x, double y) {
		return tileMap[(int)x/tileSize][(int)y/tileSize];
	}

	/**
	 * Ändrar en specifik rutas värde i rutnätet
	 * 
	 * @param col   rutans kolumnnummer
	 * @param row   rutans radnummer
	 * @param value nya värdet i rutan
	 */
	public void setTile(int col, int row, int value) {
		tileMap[col][row] = value;
	}

	/**
	 * Ändrar en specifik rutas värde i rutnätet med utgångsdpunkt från en punkt på skärmen (px)
	 * 
	 * @param x punktens position i x-led
	 * @param y punktens position i y-led
	 * @param value nya värdet i rutan
	 */
	public void setTileFromPx(double x, double y, int value) {
		tileMap[(int)x/tileSize][(int)y/tileSize] = value;
	}
}

